using System;
using System.Collections;
using System.IO;
using DynaPDF;

namespace examples
{
	class Class1
	{
	   private static CPDF m_PDF;

      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write(string.Format("{0}", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage)));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      static int FontNotFoundProc(IntPtr Data, IntPtr PDFFont, IntPtr FontName, TFStyle Style, int StdFontIndex, int IsSymbolFont)
      {
         if (m_PDF.WeightFromStyle(Style) < 500)
         {
            Style  = Style & (TFStyle)0xF;
            Style |= TFStyle.fsRegular;
         }
         return m_PDF.ReplaceFont(PDFFont, "Arial", Style, true);
      }

      static int ReplaceICCProfileProc(IntPtr Data, TICCProfileType Type, int ColorSpace)
      {
         // The most important ICC profiles are available free of charge from Adobe. Just seach for "Adobe icc profiles".
         switch((int)Type)
         {
            case 1:  return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/sRGB.icc");
            case 2:  return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/ISOcoated_v2_bas.ICC"); // This is just an example CMYK profile that can be delivered with DynaPDF.
            default: return m_PDF.ReplaceICCProfile(ColorSpace, "../../../../test_files/gray.icc");
         }
      }

      static bool ConvertFile(TConformanceType Type, String InFile, String OutFile)
      {
         int retval;
         TCheckOptions convFlags = (TCheckOptions)0;

         m_PDF.CreateNewPDF(null);                         // The output file will be created later
         m_PDF.SetDocInfo(TDocumentInfo.diProducer, null); // No need to override the original producer

         // To determine how the profiles FacturX, ZUGFeRD, and XRechnung can be used, please take a look into the example conv_to_zugferd.
         switch(Type)
         {
            case TConformanceType.ctNormalize:    convFlags = TCheckOptions.coAllowDeviceSpaces;                            break; // For normalization it is not required to convert device spaces to ICC based color spaces.
            case TConformanceType.ctPDFA_1b_2005: convFlags = TCheckOptions.coDefault | TCheckOptions.coFlattenLayers;      break; // Presentations are not prohibited in PDF/A 1.
            case TConformanceType.ctPDFA_2b:
            case TConformanceType.ctPDFA_2u:      convFlags = TCheckOptions.coDefault | TCheckOptions.coDeletePresentation; break;
            case TConformanceType.ctPDFA_3b:
            case TConformanceType.ctPDFA_3u:
            case TConformanceType.ctPDFA_4:
            case TConformanceType.ctPDFA_4e:
            case TConformanceType.ctPDFA_4f:
               convFlags  = TCheckOptions.coDefault | TCheckOptions.coDeletePresentation;
               convFlags &= ~TCheckOptions.coDeleteEmbeddedFiles; // Embedded files are allowed in PDF/A 3.
               break;
         }
         // These flags require some processing time but they are very useful.
         convFlags |= TCheckOptions.coCheckImages;
         convFlags |= TCheckOptions.coRepairDamagedImages;

         if (Type != TConformanceType.ctNormalize)
         {
            // The flag ifPrepareForPDFA is required. The flag ifImportAsPage makes sure that pages will not be converted to templates.
            m_PDF.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage | TImportFlags.ifPrepareForPDFA);
            // The flag if2UseProxy reduces the memory usage. The duplicate check is optional but recommended.
            m_PDF.SetImportFlags2(TImportFlags2.if2UseProxy | TImportFlags2.if2DuplicateCheck);
         }else
         {
            m_PDF.SetImportFlags(TImportFlags.ifImportAll | TImportFlags.ifImportAsPage);
            m_PDF.SetImportFlags2(TImportFlags2.if2UseProxy | TImportFlags2.if2Normalize);
         }
         if ((retval = m_PDF.OpenImportFile(InFile, TPwdType.ptOpen, null)) < 0)
         {
            if (m_PDF.IsWrongPwd(retval))
            {
               Console.Write("PDFError File is encrypted!");
            }
            m_PDF.FreePDF();
            return false;
         }
         m_PDF.ImportPDFFile(1, 1.0, 1.0);
         m_PDF.CloseImportFile();

         // The CMYK profile is just an example profile that can be delivered with DynaPDF.
         // Note that this code requires the PDF/A Extension for DynaPDF.
         switch ((retval = m_PDF.CheckConformance(Type, convFlags, IntPtr.Zero, FontNotFoundProc, ReplaceICCProfileProc)))
         {
            case 1: m_PDF.AddOutputIntent("../../../../test_files/sRGB.icc");             break;
            case 2: m_PDF.AddOutputIntent("../../../../test_files/ISOcoated_v2_bas.ICC"); break;
            case 3: m_PDF.AddOutputIntent("../../../../test_files/gray.icc");             break;
         }
         // No fatal error occurred?
         if (m_PDF.HaveOpenDoc())
         {
            if (!m_PDF.OpenOutputFile(OutFile))
            {
               m_PDF.FreePDF();
               return false;
            }
            m_PDF.CloseFile();
            return true;
         }
         return false;
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            m_PDF = new CPDF();
            m_PDF.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));

            // Set the license key here if you have one
            // m_PDF.SetLicenseKey("");

            // Non embedded CID fonts depend usually on the availability of external cmaps.
            // So, external cmaps should be loaded if possible.
            m_PDF.SetCMapDir(Path.GetFullPath("../../../../../Resource/CMap"), TLoadCMapFlags.lcmDelayed | TLoadCMapFlags.lcmRecursive);

            String outFile = Directory.GetCurrentDirectory() + "/out.pdf";

            /* To create a ZUGFeRD invoice attach the required XML invoice here and set the conversion type to the ZUGFeRD profile that you need.

               Example (ZUGFeRD 2.1):
               m_PDF.AttachFile("c:/invoices/test/factur-x.xml", "ZUGFeRD 2.1 Rechnung", true);
               if (ConvertFile(TConformanceType.ctFacturX_Comfort, "c:/invoices/test/TestInvoice.pdf", outFile))
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", outFile);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = outFile;
                  p.Start();
               }

               The file name of the XML invoice must be factur-x.xml. If the file has another name than rename it or use AttachFileEx() instead.
            */

            if (ConvertFile(TConformanceType.ctPDFA_3b, "../../../../../dynapdf_help.pdf", outFile))
            {
               Console.Write("PDF file \"{0}\" successfully created!\n", outFile);
               System.Diagnostics.Process p = new System.Diagnostics.Process();
               p.StartInfo.FileName = outFile;
               p.Start();
            }
            m_PDF = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
		}
	}
}
